"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitydocSocket = exports.Identitydoc = void 0;
const websocket_message_1 = require("./websocket-message");
const agent_1 = require("./agent");
const connector_1 = require("./connector");
class Identitydoc {
    constructor() { }
    async init(params) {
        const webSocketUrl = params.connectionUrl;
        const apiKey = params.auth.credentials.apiKey;
        const appWebsocketUrl = `${webSocketUrl}?ApiKey=${apiKey}`;
        const agentDomain = params.agentDomain;
        this.socket = new IdentitydocSocket(appWebsocketUrl, agentDomain);
    }
}
exports.Identitydoc = Identitydoc;
class IdentitydocSocket {
    constructor(webSocketUrl, agentDomain = null) {
        this.webSocketUrl = webSocketUrl;
        this.agentDomain = agentDomain;
    }
    close() {
        if (this.webSocket) {
            this.webSocket.close();
            this.webSocket = null;
        }
    }
    async start() {
        return new Promise((resolve, reject) => {
            if (this.webSocket != null) {
                this.close();
            }
            this.webSocket = new WebSocket(this.webSocketUrl);
            this.webSocket.onopen = () => {
                console.log('WebSocket Opened');
                this.initSocketConnections();
                resolve();
            };
            this.webSocket.onerror = (e) => {
                console.log('WebSocket Error: ', e);
                reject(e);
            };
            this.webSocket.onclose = () => {
                console.log('WebSocket Close, restarting');
                this.start();
            };
            this.webSocket.onmessage = (e) => {
                const websocketMessage = JSON.parse(e.data);
                console.log('Websocket Message: ', websocketMessage);
                // Open channel response
                if ((0, websocket_message_1.isChannelResponsePacket)(websocketMessage)) {
                    const channels = websocketMessage.payload;
                    this.startIntent(channels);
                }
                else if (this.onMessage && typeof this.onMessage === 'function') {
                    // FIXME: facciamo passare CHANNEL_RESPONSE alla onMessage?
                    this.onMessage(websocketMessage);
                }
            };
        });
    }
    subscribe(fn) {
        this.onMessage = fn;
    }
    // WebSocket initialization section
    initSocketConnections() {
        this.subscribeToSocket();
    }
    subscribeToSocket() {
        const subscribeMessage = {
            action: 'subscribe'
        };
        this.webSocket.send(JSON.stringify(subscribeMessage));
    }
    // End section
    // Connector section
    startAgent() {
        const createChannelMessage = {
            action: 'channel-request'
        };
        this.webSocket.send(JSON.stringify(createChannelMessage));
    }
    sendCan(can) {
        const readCieMessage = {
            action: 'connector-request',
            payload: {
                message: connector_1.ConnectorMessageType.CIE_READ_DATA,
                data: JSON.stringify({ CAN: can })
            }
        };
        this.webSocket.send(JSON.stringify(readCieMessage));
    }
    buildIntentUrl(pub, sub) {
        return ('ddna://' +
            this.agentDomain +
            '?proto=https:&pub=' +
            sub.replace(/^http[s]?:\/\//g, '') +
            '&sub=' +
            pub.replace(/^http[s]?:\/\//g, ''));
    }
    startIntent(channels) {
        const intentUrl = this.buildIntentUrl(channels.pub, channels.sub);
        console.debug({ intentUrl });
        const iframe = document.createElement("iframe");
        iframe.style.display = "none";
        iframe.src = intentUrl;
        document.body.appendChild(iframe);
    }
    // End section
    // Agent Download section
    downloadAgent() {
        const a = document.createElement("a");
        a.href = (0, agent_1.getClientDownloadUrl)();
        a.target = "_blank";
        a.click();
    }
}
exports.IdentitydocSocket = IdentitydocSocket;
//# sourceMappingURL=data:application/json;base64,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